<?php

namespace App\Http\Controllers;

use App\Models\Harvest;
use App\Models\Plantation;
use App\Models\PlantDetail;
use App\Models\User;
use Illuminate\Http\Request;

class HarvestController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $user = auth()->user();

        // Fetch plantations based on user role
        if ($user->role == 'Tuan_Kebun') {
            $plantations = Plantation::with(['farm' => function ($query) {
                $query->withTrashed();
            }, 'plantDetail' => function ($query) {
                $query->withTrashed();
            }, 'pengait' => function ($query) {
                $query->withTrashed();
            }, 'harvest' => function ($query) {
                $query->withTrashed();
            }])->get();
        } else {
            $pengait = Plantation::where('pengait_id', $user->id)->pluck('id');
            $plantations = Plantation::whereIn('id', $pengait)
                ->with(['farm' => function ($query) {
                    $query->withTrashed();
                }, 'plantDetail' => function ($query) {
                    $query->withTrashed();
                }, 'pengait' => function ($query) {
                    $query->withTrashed();
                }, 'harvest' => function ($query) {
                    $query->withTrashed();
                }])
                ->get();
        }

        // Fetch all harvests
        $harvests = Harvest::all();

        // Calculate total overall harvest
        $totalOverallHarvest = $harvests->sum('total_weight');

        $totalHarvests = $harvests->count(); // Count all harvests

        // Format the data for the chart
        $chartData = [];
        foreach ($harvests as $harvest) {
            $chartData[] = [
                'harvest_date' => $harvest->harvest_date,
                'gred_a_quantity' => $harvest->grade_a_quantity,
                'gred_b_quantity' => $harvest->grade_b_quantity,
                'gred_c_quantity' => $harvest->grade_c_quantity,
                'gred_d_quantity' => $harvest->grade_reject_quantity,
            ];
        }

        // Pass all data to the view
        return view('Harvest.index', compact(
            'harvests',
            'plantations',
            'totalOverallHarvest',
            'totalHarvests',
            'chartData' // Include chartData
        ));
    }



    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */

    public function insert($id)
    {
        $plantation = Plantation::with(['farm' => function ($query) {
            $query->withTrashed();
        }, 'tumbesaran' => function ($query) {
            $query->withTrashed();
        }, 'akar' => function ($query) {
            $query->withTrashed();
        }, 'buah' => function ($query) {
            $query->withTrashed();
        }, 'matang' => function ($query) {
            $query->withTrashed();
        }, 'plantDetail' => function ($query) {
            $query->withTrashed();
        }, 'pengait' => function ($query) {
            $query->withTrashed();
        }, 'penanam' => function ($query) {
            $query->withTrashed();
        }])
            ->findOrFail($id);
        $harvests = Harvest::All();
        return view('Harvest.create', compact('harvests', 'plantation'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \App\Http\Requests\StoreHarvestRequest  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'plantation_id' => 'required|exists:plantations,id',
            'harvest_date' => 'required|exists:plantations,harvest_date',
            'grade_a_quantity' => 'required|integer',
            'grade_b_quantity' => 'required|integer',
            'grade_c_quantity' => 'required|integer',
            'grade_a_weight' => 'required|integer',
            'grade_b_weight' => 'required|integer',
            'grade_c_weight' => 'required|integer',
            'total_weight' => 'required|integer',
        ]);

        $plantation = Plantation::findOrFail($request->plantation_id);
        $plantation->status = 'Selesai';
        $plantation->save();

        $harvest = new Harvest();

        $harvest->plantation_id = $request->plantation_id;
        $harvest->harvest_date = $request->harvest_date;
        $harvest->grade_a_quantity = $request->grade_a_quantity;
        $harvest->grade_b_quantity = $request->grade_b_quantity;
        $harvest->grade_c_quantity = $request->grade_c_quantity;
        $harvest->grade_reject_quantity = $request->grade_reject_quantity;

        $harvest->grade_a_weight = $request->grade_a_weight;
        $harvest->grade_b_weight = $request->grade_b_weight;
        $harvest->grade_c_weight = $request->grade_c_weight;
        $harvest->grade_reject_weight = $request->grade_reject_weight;
        $harvest->total_weight = $request->total_weight;

        $harvest->grade_a_available = $request->grade_a_weight;
        $harvest->grade_b_available = $request->grade_b_weight;
        $harvest->grade_c_available = $request->grade_c_weight;
        $harvest->status = 'Kira Hasil';

        $harvest->save();
        return redirect()->route('harvest.index')->with('success', 'Maklumat Hasil Tuai Berjaya Disimpan.');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Harvest  $harvests
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $harvest = Harvest::with([
            'plantation.farm' => function ($query) {
                $query->withTrashed(); // Include soft-deleted farms
            },
            'plantation.plantDetail' => function ($query) {
                $query->withTrashed(); // Include soft-deleted plant details
                
            }
        ])->findOrFail($id);

        $chartData = [
            ['grade' => 'Grade A', 'quantity' => $harvest->grade_a_quantity],
            ['grade' => 'Grade B', 'quantity' => $harvest->grade_b_quantity],
            ['grade' => 'Grade C', 'quantity' => $harvest->grade_c_quantity],
            ['grade' => 'Grade Tolak', 'quantity' => $harvest->grade_reject_quantity],
        ];

        // Prepare data for the weight chart
        $weightChartData = [
            ['category' => 'Grade A', 'value' => $harvest->grade_a_weight],
            ['category' => 'Grade B', 'value' => $harvest->grade_b_weight],
            ['category' => 'Grade C', 'value' => $harvest->grade_c_weight],
            ['category' => 'Grade Tolak', 'value' => $harvest->grade_reject_weight],
        ];

        return view('Harvest.show', compact('harvest', 'chartData', 'weightChartData'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Harvest  $harvests
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $harvest = Harvest::with([
            'plantation.farm' => function ($query) {
                $query->withTrashed(); // Include soft-deleted farms
            },
            'plantation.plantDetail' => function ($query) {
                $query->withTrashed(); // Include soft-deleted plant details
            }
        ])->findOrFail($id);
        return view('Harvest.edit', compact('harvest'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \App\Http\Requests\UpdateHarvestRequest  $request
     * @param  \App\Models\Harvest  $harvests
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'grade_a_weight' => 'required|integer|min:0',
            'grade_b_weight' => 'required|integer|min:0',
            'grade_c_weight' => 'required|integer|min:0',
            'total_weight' => 'required|integer|min:0',
        ]);

        // Find the harvest record using the plantation ID
        $harvest = Harvest::where('plantation_id', $id)->first();

        if ($harvest) {
            // Update the quantities
            $harvest->grade_a_weight = $request->grade_a_weight;
            $harvest->grade_b_weight = $request->grade_b_weight;
            $harvest->grade_c_weight = $request->grade_c_weight;

            $harvest->grade_a_available = $request->grade_a_weight;
            $harvest->grade_b_available = $request->grade_b_weight;
            $harvest->grade_c_available = $request->grade_c_weight;

            $harvest->total_weight = $request->total_weight;

            // Save the updated record
            $harvest->save();

            return redirect()->route('harvest.index')->with('success', 'Maklumat Hasil Tuai Berjaya Dikemaskini.');
        } else {
            return redirect()->route('harvest.index')->with('error', 'Rekod tidak dijumpai.');
        }
    }


    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Harvest  $harvests
     * @return \Illuminate\Http\Response
     */
    public function destroy(Harvest $harvests)
    {
        //
    }
}
