<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\Plantation;
use App\Models\User;
use App\Models\Damage;
use App\Models\Farm;
use App\Models\FertilizerDetail;
use App\Models\Sale;
use Illuminate\Support\Facades\Storage;
use Illuminate\Http\Request;

class DashboardController extends Controller
{
    public function index()
    {
        // Fetch the authenticated user
        $user = auth()->user();

        // Fetch all plantations with their fertilization dates and related data
        $plantations = Plantation::with([
            'farm' => function ($query) {
                $query->withTrashed();
            },
            'tumbesaran' => function ($query) {
                $query->withTrashed()->with([
                    'fertilizerDetail' => function ($query) {
                        $query->withTrashed();
                    }
                ]);
            },
            'akar' => function ($query) {
                $query->withTrashed();
            },
            'buah' => function ($query) {
                $query->withTrashed();
            },
            'matang' => function ($query) {
                $query->withTrashed();
            },
            'plantDetail' => function ($query) {
                $query->withTrashed();
            },
            'pengait' => function ($query) {
                $query->withTrashed();
            },
            'penanam' => function ($query) {
                $query->withTrashed();
            }
        ])->get();

        // Prepare events for the calendar
        $events = [];
        // Initialize counts for Penanam and Pengait
        $plantCount = 0;
        $harvestCount = 0;

        foreach ($plantations as $plantation) {
            // Check if the user is Tuan_Kebun, Penanam, or Pengait
            if (
                $user->role === 'Tuan_Kebun' || $user->role === 'admin' ||
                ($user->role === 'Pekerja' && $user->job_type === 'Penanam' && $plantation->penanam_id === $user->id) ||
                ($user->role === 'Pekerja' && $user->job_type === 'Pengait' && $plantation->pengait_id === $user->id)
            ) {
                // Add Plant Date
                if ($plantation->plant_date) {
                    $events[] = [
                        'title' => 'Tanam - ' . $plantation->farm->name,
                        'start' => $plantation->plant_date,
                        'color' => '#6c757d', // Gray color for Plant Date
                        'extendedProps' => [
                            'type' => 'Tanam',
                            'farmName' => $plantation->farm->name,
                            'plantDetail' => $plantation->plantDetail->plant_type,
                            'penanam' => $plantation->penanam->name,
                        ],
                    ];
                    // Count plant_date for Penanam
                    if ($user->role === 'Pekerja' && $user->job_type === 'Penanam' && $plantation->penanam_id === $user->id) {
                        $plantCount++;
                    }
                }

                // Add Harvest Date
                if ($plantation->harvest_date) {
                    $events[] = [
                        'title' => 'Tuai - ' . $plantation->farm->name,
                        'start' => $plantation->harvest_date,
                        'color' => '#6610f2', // Purple color for Harvest Date
                        'extendedProps' => [
                            'type' => 'Tuai',
                            'farmName' => $plantation->farm->name,
                            'plantDetail' => $plantation->plantDetail->plant_type,
                            'pengait' => $plantation->pengait->name,
                        ],
                    ];
                     // Count harvest_date for Pengait
                if ($user->role === 'Pekerja' && $user->job_type === 'Pengait' && $plantation->pengait_id === $user->id) {
                    $harvestCount++;
                }
                    
                }

                // Add Tumbesaran Dates
                $tumbesaranDates = $this->cleanAndDecodeDates($plantation->tumbesaran_date);
                foreach ($tumbesaranDates as $date) {
                    $events[] = [
                        'title' => 'Tumbesaran - ' . $plantation->farm->name,
                        'start' => $date,
                        'color' => '#28a745', // Green color for Tumbesaran
                        'extendedProps' => [
                            'type' => 'Tumbesaran',
                            'farmName' => $plantation->farm->name,
                            'plantDetail' => $plantation->plantDetail->plant_type,
                            'penanam' => $plantation->penanam->name,
                            'pengait' => $plantation->pengait->name,
                            'fertilizer_name' => $plantation->tumbesaran->fertilizerDetail->fertilizer_name ?? 'N/A',
                            'fertilizer_image' => $plantation->tumbesaran->fertilizerDetail->image ? Storage::url($plantation->tumbesaran->fertilizerDetail->image) : null,
                            'amount' => $plantation->tumbesaran->amount ?? 'N/A',
                            'amount_type' => $plantation->tumbesaran->amount_type ?? 'N/A',
                        ],
                    ];
                }

                // Add Akar Dates
                $akarDates = $this->cleanAndDecodeDates($plantation->akar_date);
                foreach ($akarDates as $date) {
                    $events[] = [
                        'title' => 'Akar - ' . $plantation->farm->name,
                        'start' => $date,
                        'color' => '#17a2b8', // Blue color for Akar
                        'extendedProps' => [
                            'type' => 'Akar',
                            'farmName' => $plantation->farm->name,
                            'plantDetail' => $plantation->plantDetail->plant_type,
                            'penanam' => $plantation->penanam->name,
                            'pengait' => $plantation->pengait->name,
                            'fertilizer_name' => $plantation->akar->fertilizerDetail->fertilizer_name ?? 'N/A',
                            'fertilizer_image' => $plantation->akar->fertilizerDetail->image ? Storage::url($plantation->akar->fertilizerDetail->image) : null,
                            'amount' => $plantation->akar->amount ?? 'N/A',
                            'amount_type' => $plantation->akar->amount_type ?? 'N/A',
                        ],
                    ];
                    
                }

                // Add Buah Dates
                $buahDates = $this->cleanAndDecodeDates($plantation->buah_date);
                foreach ($buahDates as $date) {
                    $events[] = [
                        'title' => 'Buah - ' . $plantation->farm->name,
                        'start' => $date,
                        'color' => '#ffc107', // Yellow color for Buah
                        'extendedProps' => [
                            'type' => 'Buah',
                            'farmName' => $plantation->farm->name,
                            'plantDetail' => $plantation->plantDetail->plant_type,
                            'penanam' => $plantation->penanam->name,
                            'pengait' => $plantation->pengait->name,
                            'fertilizer_name' => $plantation->buah->fertilizerDetail->fertilizer_name ?? 'N/A',
                            'fertilizer_image' => $plantation->buah->fertilizerDetail->image ? Storage::url($plantation->buah->fertilizerDetail->image) : null,
                            'amount' => $plantation->buah->amount ?? 'N/A',
                            'amount_type' => $plantation->buah->amount_type ?? 'N/A',
                        ],
                    ];
                }

                // Add Matang Dates
                $matangDates = $this->cleanAndDecodeDates($plantation->matang_date);
                foreach ($matangDates as $date) {
                    $events[] = [
                        'title' => 'Matang - ' . $plantation->farm->name,
                        'start' => $date,
                        'color' => '#dc3545', // Red color for Matang
                        'extendedProps' => [
                            'type' => 'Matang',
                            'farmName' => $plantation->farm->name,
                            'plantDetail' => $plantation->plantDetail->plant_type,
                            'penanam' => $plantation->penanam->name,
                            'pengait' => $plantation->pengait->name,
                            'fertilizer_name' => $plantation->matang->fertilizerDetail->fertilizer_name ?? 'N/A',
                            'fertilizer_image' => $plantation->matang->fertilizerDetail->image ? Storage::url($plantation->matang->fertilizerDetail->image) : null,
                            'amount' => $plantation->matang->amount ?? 'N/A',
                            'amount_type' => $plantation->matang->amount_type ?? 'N/A',
                        ],
                    ];
                }
            }
        }

        // Fetch counts for farms, fertilizer details, and users
        $totalFarms = Farm::count();
        $totalFertilizerDetails = FertilizerDetail::count();
        $totalUsers = User::count();

        // Fetch sales data from the database
        $sales = Sale::select(
            'created_at',
            'grade_a_sale',
            'grade_b_sale',
            'grade_c_sale'
        )->get();

        // Prepare data for the chart
        $chartData = $sales->map(function ($sale) {
            return [
                'year' => date('Y', strtotime($sale->created_at)), // Extract year from created_at
                'grade_a_sale' => $sale->grade_a_sale,
                'grade_b_sale' => $sale->grade_b_sale,
                'grade_c_sale' => $sale->grade_c_sale,
            ];
        });

        // Fetch damage data from the database
        $damages = Damage::all();

        // Count the occurrences of each damage category
        $damageCategories = [
            'Haiwan Perosak' => $damages->where('damage_category', 'Haiwan Perosak')->count(),
            'Penyakit' => $damages->where('damage_category', 'Penyakit')->count(),
            'Lain-lain' => $damages->where('damage_category', 'Lain-lain')->count(),
        ];
        
        // Count the number of complaints (aduan) for the authenticated user
        $jumlahAduan = Damage::where('worker_id', $user->id)->count();

        // Pass the events, counts, and chart data to the view
        return view('dashboard', compact('events', 'user', 'totalFarms', 'totalFertilizerDetails', 'totalUsers', 'chartData', 'damageCategories','plantCount', 'harvestCount','jumlahAduan'));
    }

    private function cleanAndDecodeDates($dateString)
    {
        // Remove extra quotes and trim the string
        $cleanedString = trim($dateString, '"');

        // Split the string into an array of dates
        $datesArray = explode(',', $cleanedString);

        // Trim each date to remove any extra spaces
        $datesArray = array_map('trim', $datesArray);

        return $datesArray;
    }
}
