<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\Plantation;
use App\Models\User;
use App\Models\Farm;
use App\Models\PlantDetail;
use App\Models\Fertilizer;
use App\Models\Category;
use App\Models\FertilizerCategory;
use App\Models\FertilizerDetail;
use Illuminate\Support\Facades\Storage;
use Illuminate\Http\Request;
use PhpParser\Node\Expr\New_;

class PlantationController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */

     public function index()
     {
         $user = auth()->user();
         $plantations = [];
         $filteredPlantations = []; // For table display
         $farms = Farm::withTrashed()->get(); // Include soft-deleted farms
     
         if ($user->role == 'Tuan_Kebun') {
             // Fetch all plantations for Tuan Kebun (including 'Selesai')
             $plantations = Plantation::with([
                 'farm' => function ($query) {
                     $query->withTrashed();
                 }, 
                 'tumbesaran' => function ($query) {
                     $query->withTrashed();
                 }, 
                 'akar' => function ($query) {
                     $query->withTrashed();
                 }, 
                 'buah' => function ($query) {
                     $query->withTrashed();
                 }, 
                 'matang' => function ($query) {
                     $query->withTrashed();
                 }, 
                 'plantDetail' => function ($query) {
                     $query->withTrashed();
                 }, 
                 'pengait' => function ($query) {
                     $query->withTrashed();
                 }, 
                 'penanam' => function ($query) {
                     $query->withTrashed();
                 }
             ])->get();
     
             // Filter out plantations with status 'Selesai' for table display
             $filteredPlantations = $plantations->where('status', '!=', 'Selesai');
         } else {
             // For Penanam and Pengait, fetch only assigned plantations with status != 'Selesai'
             $penanam = Plantation::where('penanam_id', $user->id)
                 ->where('status', '!=', 'Selesai')
                 ->pluck('id');
             $pengait = Plantation::where('pengait_id', $user->id)
                 ->where('status', '!=', 'Selesai')
                 ->pluck('id');
     
             // Combine the IDs and retrieve the plantations
             $plantationIds = $penanam->merge($pengait);
             $plantations = Plantation::whereIn('id', $plantationIds)
                 ->with([
                     'farm' => function ($query) {
                         $query->withTrashed();
                     }, 
                     'tumbesaran' => function ($query) {
                         $query->withTrashed();
                     }, 
                     'akar' => function ($query) {
                         $query->withTrashed();
                     }, 
                     'buah' => function ($query) {
                         $query->withTrashed();
                     }, 
                     'matang' => function ($query) {
                         $query->withTrashed();
                     }, 
                     'plantDetail' => function ($query) {
                         $query->withTrashed();
                     }, 
                     'pengait' => function ($query) {
                         $query->withTrashed();
                     }, 
                     'penanam' => function ($query) {
                         $query->withTrashed();
                     }
                 ])
                 ->get();
     
             // For Penanam and Pengait, filteredPlantations is the same as plantations
             // because the initial query already excludes 'Selesai' plantations
             $filteredPlantations = $plantations;
         }
     
         // Calculate counts for plantations
         $jumlahTanaman = $plantations->count(); // Includes all plantations for Tuan Kebun
         $baru = $plantations->where('status', 'Baru')->count();
         $tanam = $plantations->where('status', 'Tanam')->count();
     
         // Get the IDs of farms that are assigned to plantations with status != 'Selesai'
         $assignedFarmIds = Plantation::where('status', '!=', 'Selesai')->pluck('farm_id')->toArray();
     
         // Filter out farms that are assigned to active plantations (status != 'Selesai')
         $availableFarms = $farms->whereNotIn('id', $assignedFarmIds);
     
         // Get farms with plantations that have status 'Selesai'
         $selesaiFarms = Farm::whereHas('plantations', function ($query) {
             $query->where('status', 'Selesai');
         })->get();
     
         // Filter out farms from $selesaiFarms that are also in $assignedFarmIds
         $selesaiFarms = $selesaiFarms->whereNotIn('id', $assignedFarmIds);
     
         // Merge available farms and filtered $selesaiFarms
         $kebunKosong = $availableFarms->merge($selesaiFarms)->unique('id')->count();
     
         return view('plantation.index', compact(
             'filteredPlantations', // Use this for table display
             'jumlahTanaman',
             'baru',
             'tanam',
             'kebunKosong'
         ));
     }

    // display dekat jadual semasa
    public function calendar()
    {
        // Fetch the authenticated user
        $user = auth()->user();

        // Fetch all plantations with their fertilization dates and related data
        $plantations = Plantation::with([
            'farm' => function ($query) {
                $query->withTrashed();
            },
            'tumbesaran' => function ($query) {
                $query->withTrashed()->with([
                    'fertilizerDetail' => function ($query) {
                        $query->withTrashed();
                    }
                ]);
            },
            'akar' => function ($query) {
                $query->withTrashed();
            },
            'buah' => function ($query) {
                $query->withTrashed();
            },
            'matang' => function ($query) {
                $query->withTrashed();
            },
            'plantDetail' => function ($query) {
                $query->withTrashed();
            },
            'pengait' => function ($query) {
                $query->withTrashed();
            },
            'penanam' => function ($query) {
                $query->withTrashed();
            }
        ])
            ->orderBy('created_at', 'desc')
            ->get();

        // Prepare events for the calendar
        $events = [];

        foreach ($plantations as $plantation) {
            // Check if the user is Tuan_Kebun, Penanam, or Pengait
            if (
                $user->role === 'Tuan_Kebun' || $user->role === 'admin' ||
                ($user->role === 'Pekerja' && $user->job_type === 'Penanam' && $plantation->penanam_id === $user->id) ||
                ($user->role === 'Pekerja' && $user->job_type === 'Pengait' && $plantation->pengait_id === $user->id)
            ) {

                // Add Plant Date
                if ($plantation->plant_date) {
                    $events[] = [
                        'title' => 'Tanam - ' . $plantation->farm->name,
                        'start' => $plantation->plant_date,
                        'color' => '#6c757d', // Gray color for Plant Date
                        'extendedProps' => [
                            'type' => 'Tanam',
                            'farmName' => $plantation->farm->name,
                            'plantDetail' => $plantation->plantDetail->plant_type,
                            'penanam' => $plantation->penanam->name,

                        ],
                    ];
                }

                // Add Harvest Date
                if ($plantation->harvest_date) {
                    $events[] = [
                        'title' => 'Tuai - ' . $plantation->farm->name,
                        'start' => $plantation->harvest_date,
                        'color' => '#6610f2', // Purple color for Harvest Date
                        'extendedProps' => [
                            'type' => 'Tuai',
                            'farmName' => $plantation->farm->name,
                            'plantDetail' => $plantation->plantDetail->plant_type,

                            'pengait' => $plantation->pengait->name,
                        ],
                    ];
                }

                // Add Tumbesaran Dates
                $tumbesaranDates = $this->cleanAndDecodeDates($plantation->tumbesaran_date);
                foreach ($tumbesaranDates as $date) {
                    $events[] = [
                        'title' => 'Tumbesaran - ' . $plantation->farm->name,
                        'start' => $date,
                        'color' => '#28a745', // Green color for Tumbesaran
                        'extendedProps' => [
                            'type' => 'Tumbesaran',
                            'farmName' => $plantation->farm->name,
                            'plantDetail' => $plantation->plantDetail->plant_type,
                            'penanam' => $plantation->penanam->name,
                            'pengait' => $plantation->pengait->name,
                            'fertilizer_name' => $plantation->tumbesaran->fertilizerDetail->fertilizer_name ?? 'N/A',
                            'fertilizer_image' => $plantation->tumbesaran->fertilizerDetail->image ? Storage::url($plantation->tumbesaran->fertilizerDetail->image) : null,
                            'amount' => $plantation->tumbesaran->amount ?? 'N/A',
                            'amount_type' => $plantation->tumbesaran->amount_type ?? 'N/A',
                        ],
                    ];
                }

                // Add Akar Dates
                $akarDates = $this->cleanAndDecodeDates($plantation->akar_date);
                foreach ($akarDates as $date) {
                    $events[] = [
                        'title' => 'Akar - ' . $plantation->farm->name,
                        'start' => $date,
                        'color' => '#17a2b8', // Blue color for Akar
                        'extendedProps' => [
                            'type' => 'Akar',
                            'farmName' => $plantation->farm->name,
                            'plantDetail' => $plantation->plantDetail->plant_type,
                            'penanam' => $plantation->penanam->name,
                            'pengait' => $plantation->pengait->name,
                            'fertilizer_name' => $plantation->akar->fertilizerDetail->fertilizer_name ?? 'N/A',
                            'fertilizer_image' => $plantation->akar->fertilizerDetail->image ? Storage::url($plantation->akar->fertilizerDetail->image) : null,
                            'amount' => $plantation->akar->amount ?? 'N/A',
                            'amount_type' => $plantation->akar->amount_type ?? 'N/A',
                        ],
                    ];
                }

                // Add Buah Dates
                $buahDates = $this->cleanAndDecodeDates($plantation->buah_date);
                foreach ($buahDates as $date) {
                    $events[] = [
                        'title' => 'Buah - ' . $plantation->farm->name,
                        'start' => $date,
                        'color' => '#ffc107', // Yellow color for Buah
                        'extendedProps' => [
                            'type' => 'Buah',
                            'farmName' => $plantation->farm->name,
                            'plantDetail' => $plantation->plantDetail->plant_type,
                            'penanam' => $plantation->penanam->name,
                            'pengait' => $plantation->pengait->name,
                            'fertilizer_name' => $plantation->buah->fertilizerDetail->fertilizer_name ?? 'N/A',
                            'fertilizer_image' => $plantation->buah->fertilizerDetail->image ? Storage::url($plantation->buah->fertilizerDetail->image) : null,
                            'amount' => $plantation->buah->amount ?? 'N/A',
                            'amount_type' => $plantation->buah->amount_type ?? 'N/A',
                        ],
                    ];
                }

                // Add Matang Dates
                $matangDates = $this->cleanAndDecodeDates($plantation->matang_date);
                foreach ($matangDates as $date) {
                    $events[] = [
                        'title' => 'Matang - ' . $plantation->farm->name,
                        'start' => $date,
                        'color' => '#dc3545', // Red color for Matang
                        'extendedProps' => [
                            'type' => 'Matang',
                            'farmName' => $plantation->farm->name,
                            'plantDetail' => $plantation->plantDetail->plant_type,
                            'penanam' => $plantation->penanam->name,
                            'pengait' => $plantation->pengait->name,
                            'fertilizer_name' => $plantation->matang->fertilizerDetail->fertilizer_name ?? 'N/A',
                            'fertilizer_image' => $plantation->matang->fertilizerDetail->image ? Storage::url($plantation->matang->fertilizerDetail->image) : null,
                            'amount' => $plantation->matang->amount ?? 'N/A',
                            'amount_type' => $plantation->matang->amount_type ?? 'N/A',
                        ],
                    ];
                }
            }
        }

        // Pass the events to the view
        return view('plantation.schedule', compact('events'));
    }

    /**
     * Clean and decode a comma-separated date string into an array of dates.
     *
     * @param string $dateString
     * @return array
     */
    private function cleanAndDecodeDates($dateString)
    {
        // Remove extra quotes and trim the string
        $cleanedString = trim($dateString, '"');

        // Split the string into an array of dates
        $datesArray = explode(',', $cleanedString);

        // Trim each date to remove any extra spaces
        $datesArray = array_map('trim', $datesArray);

        return $datesArray;
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */


     public function create()
     {
         // Retrieve all farms
         $farms = Farm::with('category')->get();
     
         // Get the IDs of farms that are assigned to plantations with a status other than 'Selesai'
         $assignedFarmIds = Plantation::where('status', '!=', 'Selesai')->pluck('farm_id')->toArray();
     
         // Filter out the farms that are assigned to active plantations (status != 'Selesai')
         $availableFarms = $farms->whereNotIn('id', $assignedFarmIds);
     
         // Get farms with plantations that have status 'Selesai'
         $selesaiFarms = Farm::whereHas('plantations', function ($query) {
             $query->where('status', 'Selesai');
         })->get();
     
         // Filter out farms from $selesaiFarms that are also in $assignedFarmIds
         $selesaiFarms = $selesaiFarms->whereNotIn('id', $assignedFarmIds);
     
         // Merge available farms and filtered $selesaiFarms
         $allFarms = $availableFarms->merge($selesaiFarms);
     
         // Ensure no duplicates in the final list
         $allFarms = $allFarms->unique('id');
     
         $categories = Category::all();
         $fertilizerCategories = FertilizerCategory::all();
     
         // Retrieve plant details and group by category
         $plantDetails = PlantDetail::all()->groupBy('category_id');
     
         // Retrieve fertilizers grouped by plant detail
         $fertilizers = Fertilizer::with([
             'fertilizerDetail.fertilizerCategory', // Include related category
             'fertilizerDetail'
         ])->get()->groupBy('plant_detail_id');
     
         $users = User::where('role', 'Pekerja')
             ->whereIn('job_type', ['Penanam', 'Pengait'])
             ->get();
     
         // Pass data to the view
         return view('plantation.create', compact('farms', 'plantDetails', 'fertilizers', 'users', 'categories', 'fertilizerCategories', 'allFarms'));
     }

    public function checkDate(Request $request)
    {
        $plant_date = $request->input('plant_date');
        $harvest_date = $request->input('harvest_date');

        // Check if the plant_date is already assigned to a Penanam
        $assignedPenanam = Plantation::where('plant_date', $plant_date)->first();
        // Check if the harvest_date is already assigned to a Pengait
        $assignedPengait = Plantation::where('harvest_date', $harvest_date)->first();

        return response()->json([
            'assignedPenanam' => $assignedPenanam ? $assignedPenanam->penanam->name : null,
            'assignedPengait' => $assignedPengait ? $assignedPengait->pengait->name : null,
        ]);
    }

    public function filterUsers(Request $request)
    {
        $plant_date = $request->input('plant_date');
        $harvest_date = $request->input('harvest_date');

        // Retrieve all users
        $users = User::where('role', 'Pekerja')->whereIn('job_type', ['Penanam', 'Pengait'])->get();

        // Initialize user collections
        $penanamUsers = $users->where('job_type', 'Penanam');
        $pengaitUsers = $users->where('job_type', 'Pengait');

        // Filter out Penanam users who are already assigned on the selected plant_date
        if ($plant_date) {
            $assignedPenanamIds = Plantation::where('plant_date', $plant_date)->pluck('penanam_id')->toArray();
            $penanamUsers = $penanamUsers->whereNotIn('id', $assignedPenanamIds);
        }

        // Filter out Pengait users who are already assigned on the selected harvest_date
        if ($harvest_date) {
            $assignedPengaitIds = Plantation::where('harvest_date', $harvest_date)->pluck('pengait_id')->toArray();
            $pengaitUsers = $pengaitUsers->whereNotIn('id', $assignedPengaitIds);
        }

        return response()->json([
            'penanamUsers' => $penanamUsers,
            'pengaitUsers' => $pengaitUsers,
        ]);
    }


    /**
     * Store a newly created resource in storage.
     *
     * @param  \App\Http\Requests\StorePlantationRequest  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'farm_id' => 'required|exists:farms,id',
            'tumbesaran_id' => 'required|exists:fertilizer_details,id',
            'tumbesaran_date' => 'array|date',
            'akar_id' => 'required|exists:fertilizer_details,id',
            'akar_date' => 'array|date',
            'buah_id' => 'required|exists:fertilizer_details,id',
            'buah_date' => 'array|date',
            'matang_id' => 'required|exists:fertilizer_details,id',
            'matang_date' => 'array|date',
            'harvest_date' => 'nullable|date',
            'plant_date' => 'nullable|date',
            'quantity_plant' => 'nullable|integer',
            'status' => 'nullable|string',
            'penanam_id' => 'required|exists:users,id',
            'pengait_id' => 'required|exists:users,id',

        ]);

        $plantation = new Plantation();

        $plantation->farm_id = $request->farm_id;
        $plantation->plant_detail_id = $request->plant_type;
        $plantation->tumbesaran_id = $request->tumbesaran_id;
        $plantation->akar_id = $request->akar_id;
        $plantation->buah_id = $request->buah_id;
        $plantation->matang_id = $request->matang_id;
        $plantation->harvest_date = $request->harvest_date;
        $plantation->plant_date = $request->plant_date;
        $plantation->quantity_plant = $request->quantity_plant;
        $plantation->status = 'Baru';
        $plantation->penanam_id = $request->penanam_id;
        $plantation->pengait_id = $request->pengait_id;

        // Plantation::create($request->all());


        // Directly assign the arrays to the plantation model
        $plantation->tumbesaran_date = $request->tumbesaran_dates;
        $plantation->akar_date = $request->akar_dates;
        $plantation->buah_date = $request->buah_dates;
        $plantation->matang_date = $request->matang_dates;

        $plantation->save();

        return redirect()->route('plantation.index')->with('success', 'Tanaman Semasa Berjaya Disimpan');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Plantation  $plantation
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {

        $plantations = Plantation::with([
            'farm' => function ($query) {
                $query->withTrashed();
            },
            'tumbesaran' => function ($query) {
                $query->withTrashed()->with([
                    'fertilizerDetail' => function ($query) {
                        $query->withTrashed();
                    }
                ]);
            },
            'akar' => function ($query) {
                $query->withTrashed()->with([
                    'fertilizerDetail' => function ($query) {
                        $query->withTrashed();
                    }
                ]);
            },
            'buah' => function ($query) {
                $query->withTrashed()->with([
                    'fertilizerDetail' => function ($query) {
                        $query->withTrashed();
                    }
                ]);
            },
            'matang' => function ($query) {
                $query->withTrashed()->with([
                    'fertilizerDetail' => function ($query) {
                        $query->withTrashed();
                    }
                ]);
            },
            'plantDetail' => function ($query) {
                $query->withTrashed();
            },
            'pengait',
            'penanam'
        ])->findOrFail($id);
        
        return view('plantation.show', compact('plantations'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Plantation  $plantation
     * @return \Illuminate\Http\Response
     */

    public function edit($id)
    {
        // Fetch the plantation with related data
        $plantation = Plantation::with([
            'farm' => function ($query) {
                $query->withTrashed();
            },
            'tumbesaran' => function ($query) {
                $query->withTrashed()->with([
                    'fertilizerDetail' => function ($query) {
                        $query->withTrashed();
                    }
                ]);
            },
            'akar' => function ($query) {
                $query->withTrashed()->with([
                    'fertilizerDetail' => function ($query) {
                        $query->withTrashed();
                    }
                ]);
            },
            'buah' => function ($query) {
                $query->withTrashed()->with([
                    'fertilizerDetail' => function ($query) {
                        $query->withTrashed();
                    }
                ]);
            },
            'matang' => function ($query) {
                $query->withTrashed()->with([
                    'fertilizerDetail' => function ($query) {
                        $query->withTrashed();
                    }
                ]);
            },
            'plantDetail' => function ($query) {
                $query->withTrashed();
            },
            'pengait',
            'penanam'
        ])->findOrFail($id);
    

        // Fetch users with the role 'Pekerja' and job types 'Penanam' or 'Pengait'
        $users = User::where('role', 'Pekerja')
            ->whereIn('job_type', ['Penanam', 'Pengait'])
            ->get();

        // Fetch all farms
        $farms = Farm::all();

        // Fetch plant details grouped by category_id
        $plantDetails = PlantDetail::all()->groupBy('category_id');

        // Fetch all fertilizers grouped by plant_detail_id
        $fertilizers = Fertilizer::with([
            'fertilizerDetail.fertilizerCategory', // Include related category
            'fertilizerDetail'
        ])
            ->get()
            ->groupBy('plant_detail_id'); // Group by plant_detail_id

        // Fetch all fertilizer categories
        $fertilizerCategories = FertilizerCategory::all();

        // Pass the data to the view
        return view('plantation.edit', compact(
            'plantation',
            'users',
            'plantDetails',
            'fertilizers',
            'fertilizerCategories',
            'farms'
        ));
    }


    public function update(Request $request, $id)
    {

        // Validate the request data
        $request->validate([
            'plant_detail_id' => 'required|exists:plant_details,id',
            'tumbesaran_id' => 'required|exists:fertilizer_details,id',
            'akar_id' => 'required|exists:fertilizer_details,id',
            'buah_id' => 'required|exists:fertilizer_details,id',
            'matang_id' => 'required|exists:fertilizer_details,id',
            'penanam_id' => 'required|exists:users,id',
            'pengait_id' => 'required|exists:users,id',
            'plant_date' => 'required|date',
            'harvest_date' => 'required|date|after_or_equal:plant_date',
            'quantity_plant' => 'required|integer|min:1',
            'tumbesaran_date' => 'nullable|string',
            'akar_date' => 'nullable|string',
            'buah_date' => 'nullable|string',
            'matang_date' => 'nullable|string',

        ]);
        // dd($request->all());
        // Find the plantation record
        $plantation = Plantation::findOrFail($id);


        // Update the plantation details
        $plantation->update($request->only([
            'plant_detail_id',
            'tumbesaran_id',
            'akar_id',
            'buah_id',
            'matang_id',
            'penanam_id',
            'pengait_id',
            'plant_date',
            'harvest_date',
            'quantity_plant',
            'tumbesaran_date',
            'akar_date',
            'buah_date',
            'matang_date'
        ]));

        // dd($plantation->toArray());



        // Redirect back with a success message
        return redirect()->route('plantation.index')->with('success', "Maklumat Tanaman Semasa{$plantation->farm->name} Berjaya Dikemaskini");
    }

    public function tanam($id)
    {
        $plantation = Plantation::findOrFail($id);

        // Check the current status
        if ($plantation->status === 'Baru') {
            $plantation->status = 'Proses Tanam';
            $plantation->save();
            return redirect()->route('plantation.index')->with('success', 'Penanaman Dalam Proses.');
        }

        return redirect()->route('plantation.index')->withErrors('Status Penanaman Tidak Dapat Dikemaskini');
    }

    public function kemaskiniTanam(Request $request, $id)
    {
        $request->validate([
            'proof_tanam' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        $plantation = Plantation::findOrFail($id);

        $plantation->status = 'Tanam';

        // Handle PDF upload if a new file is provided
        if ($request->hasFile('image_tanam')) {
            // Delete the old PDF file if it exists
            if ($plantation->image_tanam) {
                Storage::delete($plantation->image_tanam);
            }
            $originalFileName = $request->file('image_tanam')->getClientOriginalName();
            $imagePath = $request->file('image_tanam')->storeAs('public/banner', $originalFileName);
            $plantation->image_tanam = $imagePath; // Save the new path to the database
        }

        $plantation->save();

        return redirect()->route('plantation.index')->with('success', 'Tanaman Semasa Berjaya Ditanam');
    }

    public function tuai($id)
    {
        $plantation = Plantation::findOrFail($id);

        // Check the current status
        if ($plantation->status === 'Tanam') {
            $plantation->status = 'Proses Tuai';
            $plantation->save();
            return redirect()->route('plantation.index')->with('success', 'Penuaian Hasil Kebun Dalam Proses.');
        }

        return redirect()->route('plantation.index')->withErrors('Status Penuaian Hasil Kebun Tidak Dapat Dikemaskini');
    }

    public function kemaskiniTuai(Request $request, $id)
    {
        $request->validate([
            'proof_tuai' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        $plantation = Plantation::findOrFail($id);

        $plantation->status = 'Tuai Hasil';

        // Handle PDF upload if a new file is provided
        if ($request->hasFile('image_kait')) {
            // Delete the old PDF file if it exists
            if ($plantation->image_kait) {
                Storage::delete($plantation->image_kait);
            }
            $originalFileName = $request->file('image_kait')->getClientOriginalName();
            $imagePath = $request->file('image_kait')->storeAs('public/banner', $originalFileName);
            $plantation->image_kait = $imagePath; // Save the new path to the database
        }

        $plantation->save();

        return redirect()->route('plantation.index')->with('success', 'Tanaman Semasa Berjaya Dituai dan Sila Kemaskini Hasil Kebun');
    }


    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Plantation  $plantation
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $plantation = Plantation::findOrFail($id);
        $plantation->delete();

        // Redirect back with a success message
        return redirect()->route('plantation.index')->with('success', 'Maklumat Tanaman Berjaya Dipadam.');
    }
}
