<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\Damage;
use App\Models\PestControl;
use App\Models\PestControlDetail;
use App\Models\Plantation;
use App\Models\Farm;
use Illuminate\Support\Facades\Storage;
use Illuminate\Http\Request;

class DamageController extends Controller
{
    public function index()
    {
        $user = auth()->user();
        $damages = [];

        if ($user->role == 'Tuan_Kebun') {
            // Fetch all damages for Tuan Kebun
            $damages = Damage::with([
                'plantation.farm' => function ($query) {
                    $query->withTrashed(); // Include soft-deleted farms
                },
                'plantation.plantDetail' => function ($query) {
                    $query->withTrashed(); // Include soft-deleted plant details
                },
                'worker'
            ])->get();
        } else {
            // Fetch damages only for the logged-in worker
            $damages = Damage::where('worker_id', $user->id)->with([
                'plantation.farm' => function ($query) {
                    $query->withTrashed(); // Include soft-deleted farms
                },
                'plantation.plantDetail' => function ($query) {
                    $query->withTrashed(); // Include soft-deleted plant details
                },
                'worker'
            ])->get();
        }

        // Calculate counts for each status
        $jumlahAduan = $damages->count();
        $aduanBaru = $damages->where('status', 'Baru')->count();
        $serahTugas = $damages->where('status', 'Serah Tugas')->count();
        $dalamKerja = $damages->where('status', 'Dalam Kerja')->count();
        $aduanSelesai = $damages->where('status', 'Selesai')->count();

        // Calculate counts for each damage category
        $damageCategories = [
            'Haiwan Perosak' => $damages->where('damage_category', 'Haiwan Perosak')->count(),
            'Penyakit' => $damages->where('damage_category', 'Penyakit')->count(),
            'Lain-lain' => $damages->where('damage_category', 'Lain-lain')->count(),
        ];
        

        return view('damage.index', compact(
            'damages',
            'jumlahAduan',
            'aduanBaru',
            'serahTugas',
            'dalamKerja',
            'aduanSelesai',
            'damageCategories'
        ));
    }

    public function create()
    {
        $damageCategories = ['Haiwan Perosak', 'Penyakit', 'Lain-lain'];
    
        // Get farms with plantations that have status 'Baru' or 'Tanam'
        $allFarms = Farm::whereHas('plantations', function ($query) {
            $query->whereIn('status', ['Baru', 'Tanam']);
        })->get();
    
        // Get plantations with their related farm and plant detail, including soft-deleted ones
        $plantations = Plantation::with([
            'farm' => function ($query) {
                $query->withTrashed();
            },
            'plantDetail' => function ($query) {
                $query->withTrashed();
            }
        ])->get();
    
        return view('damage.create', compact('plantations', 'damageCategories', 'allFarms'));
    }


    public function store(Request $request)
    {

        $request->validate([
            'damage_category' => 'required|in:Haiwan Perosak,Lain-lain,Penyakit',
            'plantation_id' => 'required|exists:plantations,id',
            'damage_description' => 'nullable|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        $damage = new Damage();
        $damage->worker_id = auth()->user()->id;
        $damage->plantation_id = $request->plantation_id;
        $damage->damage_category = $request->damage_category;
        $damage->damage_description = $request->damage_description;

        // Check if the request has an image file
        if ($request->hasFile('image') && isset($request->image)) {
            // Get the original file name
            $originalFileName = $request->file('image')->getClientOriginalName();

            // Store the image in the 'public/banner' directory with the original name
            $imagePath = $request->file('image')->storeAs('public/banner', $originalFileName);

            // Save the path to the database
            $damage->image = $imagePath; // Save the path to the database
        }
        $damage->status = 'Baru';
        $damage->save();

        return redirect()->route('damage.index')->with('success', 'Aduan Kerosakan Berjaya Disimpan!');
    }

    public function show($id)
    {
        $damage = Damage::with([
            'pestControl.pestControlDetail.pestCategory', // Load nested relationships
            'plantation.farm' => function ($query) {
                $query->withTrashed(); // Include soft-deleted farms
            },
            'plantation.plantDetail' => function ($query) {
                $query->withTrashed(); // Include soft-deleted plant details
            },
        ])->findOrFail($id);
        return view('damage.show', compact('damage'));
    }

    public function edit($id)
    {
        $damage = Damage::with([
            'plantation.farm' => function ($query) {
                $query->withTrashed(); // Include soft-deleted farms
            },
            'plantation.plantDetail' => function ($query) {
                $query->withTrashed(); // Include soft-deleted plant details
            }
        ])->findOrFail($id);
        $pestControls = PestControl::with(['pestControlDetail.pestCategory'])
            ->where('plant_detail_id', $damage->plantation->plant_detail_id)
            ->get();

        $pestControlDetail = PestControlDetail::all()->groupBy('plant_detail_id');
        $solutionCategories = ['Meracun', 'Jumpa Tuan Kebun'];
        return view('damage.edit', compact('damage', 'solutionCategories', 'pestControls', 'pestControlDetail'));
    }

    public function update(Request $request, $id)
    {
        // Validate the request
        $request->validate([
            'pest_id' => 'nullable|exists:pest_controls,id',
            'solution_category' => 'required|in:Meracun,Jumpa Tuan Kebun',
            'solution_description' => 'nullable|string',
            'assign_date' => 'required|date',
        ]);

        // Find the damage record
        $damage = Damage::findOrFail($id);

        $damage->pest_id = $request->pest_id;
        $damage->solution_category = $request->solution_category;
        $damage->solution_description = $request->solution_description;
        $damage->assign_date = $request->assign_date;
        // Set the status based on the selected solution_category
        if ($request->solution_category === 'Jumpa Tuan Kebun') {
            $damage->status = 'Selesai';
        } else {
            $damage->status = 'Serah Tugas';
        }

        $damage->save();

        return redirect()->route('damage.index')->with('success', 'Maklumat Aduan Berjaya Dikemaskini!');
    }

    public function aduan($id)
    {
        $damage = Damage::findOrFail($id);

        // Check the current status
        if ($damage->status === 'Serah Tugas') {
            $damage->status = 'Dalam Kerja';
            $damage->save();
            return redirect()->route('damage.index')->with('success', 'Membaiki Kerosakan Dalam Proses.');
        }

        return redirect()->route('damage.index')->withErrors('Status Membaiki Kerosakan Tidak Dapat Dikemaskini');
    }

    public function kemaskiniAduan(Request $request, $id)
    {
        $request->validate([
            'image_work' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        $damage = Damage::findOrFail($id);

        $damage->status = 'Selesai';

        // Handle PDF upload if a new file is provided
        if ($request->hasFile('image_work')) {
            // Delete the old PDF file if it exists
            if ($damage->image_work) {
                Storage::delete($damage->image_work);
            }
            $originalFileName = $request->file('image_work')->getClientOriginalName();
            $imagePath = $request->file('image_work')->storeAs('public/banner', $originalFileName);
            $damage->image_work = $imagePath; // Save the new path to the database
        }

        $damage->save();

        return redirect()->route('damage.index')->with('success', 'Aduan Sudah Selesai');
    }




    public function assign()
    {
        $damages = Damage::All();
        return view('damage.assign', compact('damage'));
    }
}
